#!/usr/bin/env perl

use strict;
use warnings;

# Cutecom outputs \r\n, while socat outputs \n
use open IN  => ':crlf';
use Getopt::Long;
use File::Temp ();
use bignum;

my $fancy   = 0;
my $verbose = 0;
my $help    = 0;


sub usage {
	print <<"EOF";
$0 [-v|--verbose] [--fancy] <files>
	--verbose   Log rejected lines to stderr
	--fancy     Plot the results
EOF
	exit 1;
}

GetOptions("fancy"      => \$fancy,
           "v|verbose+" => \$verbose,
           "h|?|help"   => \$help)
or usage();

usage() if $help;

my %data;
my $rejected = 0;
my $total    = 0;
# Linewise read input from files passed on the command line
while(<>) {
	# remove the newline
	chomp;
	#Input:  'time: 2047734955 value: 1214
'
	if (/^time:\s+(?<timestamp>\d+)\s+value:\s+(?<duration>\d+)$/) {
		# The hash bucket for duration will be autovivified and incremented to 1 if nonexistent
		$total++;
		$data{$+{duration}}++;
	} else {
		$rejected++;
		print STDERR "Rejected: $_\n" if $verbose;
	}
}

sub avg {
	my $acc = 0;
	$acc += $_ * $data{$_} for keys %data;
	return $acc/$total;
}

sub stddev {
	if($total == 1){
		return 0;
	}
	my $average = avg();
	my $sqtotal = 0;
	for my $val (keys %data) {
		$sqtotal += $data{$val} * (($average-$val) ** 2);
	}
	my $std = ($sqtotal / ($total-1)) ** 0.5;
	return $std;
}

#print "Info: $rejected lines failed to parse\n" if ($rejected);
printf "Occurrences Measurement\n";
for my $m (sort {$a <=> $b} keys %data) {
	printf "%11d %s\n", $data{$m}, $m;
}
my $min = (sort {$a <=> $b} keys %data)[0];
my $max = (sort {$a <=> $b} keys %data)[-1];
my $avg = avg();
my $stddev = stddev();
printf "Info: $total measurements found: min: $min, max: $max, avg: %.5f, stddev: %.5f\n", $avg, $stddev;

if ($fancy) {
	my $fh = File::Temp->new(CLEANUP=>0);
	for my $m (sort {$a <=> $b} keys %data) {
		printf $fh "%s %f\n", $m, $data{$m}/$total * 100;
	}
	my $cmd = <<"EOF";
	gnuplot --persist
	-e "
	set terminal qt;
	set grid;
	set boxwidth 10;
	set format y '%.0f%%';
    set style fill solid;
	set xlabel 'Response Time';
	set ylabel 'Occurences';
	plot '@{[$fh->filename]}' using 1:2 with boxes title 'measurement';
	pause -1;
	";
EOF
	$cmd =~ s/\n/ /g;
	print $cmd;
	system($cmd) == 0 or die "Gnuplot failed";
}
