#include <stdint.h>
#include "ezs_io.h"
#include "ezs_counter.h"
#include "ezs_stopwatch.h"
#include "ezs_test.h"

#define EZS_TIMING_START(var)          \
	ezs_watch_start(&var);

#define EZS_TIMING_RANGE_ASSERT(ctr, min, max)                                         \
	{                                                                                  \
		++ezs_tests_run;                                                               \
		if (ctr < min || ctr > max) {                                                  \
			++ezs_failed_tests;                                                        \
			ezs_printf("Timing bound failed: got %llu, expected %llu <= t <=  %llu\n", \
			        (long long unsigned) ctr,                                          \
			        (long long unsigned) min,                                          \
			        (long long unsigned) max);                                         \
		}                                                                              \
	} while (0);

#define EZS_TIMING_ASSERT(var, min, max)                                               \
	{                                                                                  \
		cyg_uint32 ctr = ezs_watch_stop(&var);                                         \
		EZS_TIMING_RANGE_ASSERT(ctr, min, max);                                        \
	} while (0);

void ezs_sanity_test(void) {
	uint32_t ezs_failed_tests = 0;
	uint32_t ezs_tests_run = 0;
	cyg_uint32 timestamp;


	ezs_printf("TESTING: ezs_(start|stop)_watch\n");
	{
		cyg_uint32 outer, inner, inner_out, outer_out;
		unsigned int i;
		unsigned int j;

		ezs_printf("Testing timekeeping functions...\n");
		for(i = 0; i < 500; i++) {
			cyg_interrupt_disable();
			outer = ezs_counter_get();
			EZS_TIMING_START(timestamp);
			inner = ezs_counter_get();
			for(j = 0; j < i; j++) {
				__asm__ volatile("nop");
			}
			inner_out = ezs_counter_get();
			timestamp = ezs_watch_stop(&timestamp);
			outer_out = ezs_counter_get();
			cyg_interrupt_enable();
 			EZS_TIMING_RANGE_ASSERT(timestamp, (inner_out - inner), (outer_out - outer));
		}
		ezs_printf("... done\n");
	}

	ezs_printf("TESTING: ezs_simulate_wcet\n");
	ezs_printf("Hint: no test should take longer than 2s\n");
	ezs_printf("Hint: if it takes longer, your implementation of ezs_simulate_wcet might be faulty\n");
	{
		unsigned int i;
		ezs_printf("Stresstesting for integer underflow (without jitter)...\n");
		for(i = 0; i < 500; i++) {
			EZS_TIMING_START(timestamp);
			ezs_simulate_wcet(0, 0);
			// Should only take a short time (the static overhead)
			EZS_TIMING_ASSERT(timestamp, 0, 500);
		}
		ezs_printf("... done\n");

		ezs_printf("Stresstesting for integer underflow (with jitter 100%%)...\n");
		for(i = 0; i < 500; i++) {
			EZS_TIMING_START(timestamp);
			ezs_simulate_wcet(0, 100);
			// Should only take a short time (the static overhead)
			EZS_TIMING_ASSERT(timestamp, 0, 500);
		}
		ezs_printf("... done\n");

		ezs_printf("Sleeping for 2000 cycles (repeatedly)...\n");
		for(i = 0; i < 500; i++) {
			EZS_TIMING_START(timestamp);
			ezs_simulate_wcet(2000, 0);
			EZS_TIMING_ASSERT(timestamp, 1900, 2050);
		}
		ezs_printf("... done\n");
	}

	ezs_printf("Total: %llu tests, %llu successful, %llu failed\n"
	          , (long long unsigned) ezs_tests_run
	          , (long long unsigned)  ezs_tests_run - ezs_failed_tests
	          , (long long unsigned)  ezs_failed_tests);
	if (ezs_failed_tests == 0) {
		ezs_printf("ALL TESTS SUCCESSFUL, you might be fine\n");
	} else {
		ezs_printf("%llu TESTS FAILED. Please check your implementation of the tested utility functions\n", (long long unsigned) ezs_failed_tests);
	}
	ezs_test_status_lcd(ezs_failed_tests);
	while(1);
	return;
}
