/*!
 * @file ezs_stopwatch.h
 * A stopwatch for time measurement and
 * WCET generation.
 *
 * \ingroup hw
 *  @brief EZS time measurement and WCET generation
 *  @author Martin Hoffmann
 */

#ifndef __STOPWATCH_H_
#define __STOPWATCH_H_
#include <cyg/infra/cyg_type.h>

#include "ezs_defs.h"

#if TT_ECOS
#include <sml/ttkernel/c_api.h>
typedef tt_ticktype tick_count_t;
#else
#include <cyg/kernel/kapi.h>
typedef cyg_tick_count_t tick_count_t;
#endif


/**
 *  Start stop watch.
 *  @param state Pointer to a state variable
 */
void ezs_watch_start(cyg_uint32* state);

/**
 *  Stop watch and return time in timer's ticks.
 *  @param state Pointer to a state variable
 *  @see ezs_watch_start
 */
cyg_uint32 ezs_watch_stop(cyg_uint32* state);

/**
 *  Waits actively for WCET simulation.
 *
 *  @param wcetticks Maximum amount of ticks to wait actively.
 *              Beware! Too small amounts of ticks may be undetectable.
 *              Thus this function may not support small granularities.
 *  @param jitter Percentage of the execution time that may randomly be skipped.
 *  			A value of 0% means that the full execution time should always be taken,
 *  			A value of 80% means that we might skip up to 80% of the intended wcet.
 *  			Values above 100 are mapped to 100.
 *
 * \see ezs_counter_get_resolution()
 */
void ezs_simulate_wcet(cyg_uint32 wcetticks, cyg_uint8 jitter);

/**
 *  Converts time from milliseconds to eCos timer ticks.
 *  @param ms time in milliseconds
 *  @return time in eCos timer ticks
 */
tick_count_t ms_to_cyg_ticks(cyg_uint32 ms);

/**
 *  Converts time from milliseconds to EZS counter ticks.
 *  @param ms time in milliseconds
 *  @return time in EZS counter ticks
 */
tick_count_t ms_to_ezs_ticks(cyg_uint32 ms);

#endif

